"use strict";
/********************************************************************************
 * Copyright (c) 2021 STMicroelectronics and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 *******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const _1 = require(".");
const chai_1 = require("chai");
const fs = require("fs");
const paths = require("path");
const common_spec_1 = require("./common.spec");
/* eslint-disable no-unused-expressions */
chai_1.use(require('chai-string'));
describe('clangd-context', function () {
    const debug = 'DEBUG';
    const release = 'RELEASE';
    let configFile;
    let project;
    this.beforeAll(common_spec_1.createTmpDir);
    this.beforeEach(function () {
        configFile = common_spec_1.tmpFile('app/.clangd', path => `CompileFlags:
    CompilationDatabase: ${paths.resolve(path, '..', debug)}
`);
        project = paths.dirname(configFile);
        common_spec_1.tmpFile(`app/${debug}/compile_commands.json`);
        common_spec_1.tmpFile(`app/${release}/compile_commands.json`);
    });
    describe('getContext', function () {
        it('get context from .clangd file', function () {
            const context = _1.getContext(configFile);
            chai_1.expect(context.name).to.equal(debug);
            chai_1.expect(context.compilationDatabase).to.equal(paths.resolve(project, debug));
        });
    });
    describe('setContext', function () {
        it('set context into .clangd file', function () {
            const contextDir = paths.resolve(project, release);
            _1.setContext(configFile, contextDir);
            const content = fs.readFileSync(configFile, 'utf8');
            chai_1.expect(content).to.have.string(contextDir);
        });
    });
    describe('setCompileFlags', function () {
        it('set removed flags', function () {
            _1.setCompileFlags(configFile, undefined, ['-a', '-b', '-c']);
            const context = _1.load(configFile);
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Remove).to.have.members(['-a', '-b', '-c']);
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Add).to.be.undefined;
        });
        it('set added flags', function () {
            _1.setCompileFlags(configFile, ['-a', '-b', '-c']);
            const context = _1.load(configFile);
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Add).to.have.members(['-a', '-b', '-c']);
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Remove).to.be.undefined;
        });
        it('set both kinds of flags', function () {
            _1.setCompileFlags(configFile, ['-a', '-b', '-c'], ['-u', '-v']);
            const context = _1.load(configFile);
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Add).to.have.members(['-a', '-b', '-c']);
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Remove).to.have.members(['-u', '-v']);
        });
    });
    describe('unsetCompileFlags', function () {
        this.beforeEach(function () {
            configFile = common_spec_1.tmpFile('app/.clangd', path => `CompileFlags:
    Add:
        - -a
        - -b
        - -c
    Remove:
        - -u
        - -v
    CompilationDatabase: ${paths.resolve(path, '..', debug)}
`);
        });
        it('unset removed flags', function () {
            _1.unsetCompileFlags(configFile, ['-u']);
            const context = _1.load(configFile);
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Remove).to.have.members(['-v']);
        });
        it('unset added flags', function () {
            _1.unsetCompileFlags(configFile, ['-b']);
            const context = _1.load(configFile);
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Add).to.have.members(['-a', '-c']);
        });
        it('unset all flags', function () {
            _1.unsetCompileFlags(configFile, ['-a', '-b', '-c', '-u', '-v']);
            const context = _1.load(configFile);
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Add).to.be.undefined;
            chai_1.expect(context === null || context === void 0 ? void 0 : context.CompileFlags.Remove).to.be.undefined;
        });
    });
    describe('workspace-aware functions', function () {
        let workspaceConfigFile;
        let workspaceConfig;
        let appProject;
        let libProject;
        this.beforeAll(function () {
            workspaceConfigFile = common_spec_1.tmpFile('.clangd-contexts', `{
"workspaceName": "Test Workspace",
"projects": [
    {
        "path": "app",
        "contextDirs": "flat"
    },
    {
        "path": "lib",
        "contextDirs": "flat"
    }
]
}`);
            common_spec_1.tmpFile(`app/${debug}/compile_commands.json`, '');
            common_spec_1.tmpFile(`app/${release}/compile_commands.json`, '');
            common_spec_1.tmpFile(`lib/${debug}/compile_commands.json`, '');
            common_spec_1.tmpFile(`lib/${release}/compile_commands.json`, '');
            workspaceConfig = _1.getClangdContextsConfig(workspaceConfigFile);
            appProject = paths.join(workspaceConfig.path, 'app');
            libProject = paths.join(workspaceConfig.path, 'lib');
        });
        it('selectContext', function () {
            _1.selectContext(workspaceConfig, release);
            const appContext = _1.loadOrCreate(appProject);
            chai_1.expect(appContext.CompileFlags.CompilationDatabase).to.have.string(release);
            const libContext = _1.loadOrCreate(libProject);
            chai_1.expect(libContext.CompileFlags.CompilationDatabase).to.have.string(release);
        });
        it('forAllProjects', function () {
            const names = [];
            _1.forAllProjects(workspaceConfig, proj => names.push(typeof proj === 'string' ? proj : proj.filePath));
            chai_1.expect(names).to.have.members([appProject, libProject]);
        });
    });
});
//# sourceMappingURL=clangd-context.spec.js.map