/*****************************************************************************
 * Copyright (c) 2022, 2025 CEA LIST.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) <vincent.lorenzo@cea.fr> - Initial API and implementation
 *  Obeo - Improvement of the checker
 *****************************************************************************/
package org.eclipse.papyrus.sirius.junit.utils.diagram.creation.semantic.checker;

import java.util.Collection;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.osgi.util.NLS;
import org.eclipse.sirius.viewpoint.DRepresentationElement;
import org.eclipse.uml2.uml.Element;
import org.junit.Assert;

/**
 * Semantic Checker for element creation.
 */
public abstract class AbstractSemanticCreationChecker implements ISemanticRepresentationElementCreationChecker {

	/** By default, when executing a creation tool, only one semantic element is created. */
	private static final int DEFAULT_CREATED_ELEMENTS_NUMBER = 1;

	/** The semantic owner of the created element */
	protected final EObject semanticOwner;
	private final EReference containmentFeature;
	private final Class<? extends Element> expectedType;

	/** The number of children in the owning feature before the creation. */
	protected final int nbChildren;

	/** The expected number of additional created elements in the {@code containmentFeature}. */
	protected int expectedCreatedElements = DEFAULT_CREATED_ELEMENTS_NUMBER;

	/** The expected number of associated semantic elements. */
	protected int expectedAssociatedElements = DEFAULT_CREATED_ELEMENTS_NUMBER;

	protected int semanticIndex = -1;

	/**
	 * Constructor.
	 *
	 * @param expectedOwner
	 *            the semantic owner
	 * @param containmentFeature
	 *            the containment feature of the created element
	 * @param expectedType
	 *            type of created element
	 */
	@SuppressWarnings("unchecked")
	public AbstractSemanticCreationChecker(final EObject expectedOwner, EReference containmentFeature) {
		this(expectedOwner, containmentFeature, (Class<? extends Element>) containmentFeature.getEReferenceType().getInstanceClass());
	}

	/**
	 * Constructor.
	 *
	 * @param expectedOwner
	 *            the semantic owner
	 * @param containmentFeature
	 *            the containment feature of the created element
	 * @param expectedType
	 *            type of created element
	 */
	public AbstractSemanticCreationChecker(final EObject expectedOwner, EReference containmentFeature, Class<? extends Element> expectedType) {
		this.semanticOwner = expectedOwner;
		this.expectedType = expectedType;
		this.containmentFeature = containmentFeature;

		Assert.assertTrue("Not multi-valued Owning feature is not yet implemented", //$NON-NLS-1$
				containmentFeature.isMany());

		if (containmentFeature != null) {
			// Feature is null if getContainmentFeature() is dynamic.
			Assert.assertTrue(NLS.bind("Invalid containment: {0} is not {1}", //$NON-NLS-1$
					semanticOwner.eClass().getName(),
					containmentFeature.getEContainingClass().getName()),
					containmentFeature.getEContainingClass().isInstance(semanticOwner));
		}

		nbChildren = getContainmentFeatureValue().size();
	}

	@Override
	public void validateRepresentationElement(DRepresentationElement createdElementRepresentation) {
		final List<EObject> semanticElements = createdElementRepresentation.getSemanticElements();
		Assert.assertEquals("Unexpected associated element", getExpectedAssociatedElements(), semanticElements.size()); //$NON-NLS-1$

		final EObject element = getSemanticElement(createdElementRepresentation);
		validateSemanticElementInstance(element);
		validateSemanticOwner(element);
	}

	/**
	 * Extracts the semantic element from provided view.
	 * <p>
	 * When semanticIndex is 0 or positive, use the SemanticElements
	 * otherwise use target property.
	 * </p>
	 *
	 * @param view
	 *            diagram element to evaluate
	 * @return
	 */
	protected EObject getSemanticElement(DRepresentationElement view) {
		if (semanticIndex < 0) {
			return view.getTarget();
		}
		return view.getSemanticElements().get(semanticIndex);
	}


	/**
	 * Validates the containment of the created element.
	 *
	 * @param semanticElement
	 *            the created element
	 */
	protected void validateSemanticOwner(final EObject semanticElement) {
		Assert.assertEquals("Semantic owner does not contain created element.", //$NON-NLS-1$
				semanticOwner, semanticElement.eContainer());
		Collection<?> values = getContainmentFeatureValue();
		Assert.assertTrue("Created element is not owned by expected feature.", //$NON-NLS-1$
				values.contains(semanticElement));
		Assert.assertEquals("Unexpected additional elements after creation.", //$NON-NLS-1$
				getNumberOfExpectedCreatedElement(), values.size() - nbChildren);
	}

	/**
	 * Validates the type of the created element.
	 *
	 * @param createdElement
	 *            the created element
	 */
	protected void validateSemanticElementInstance(EObject semanticElement) {
		// Cannot use assertEquals: inheritance is valid.
		Assert.assertTrue(NLS.bind("Unexpected class for created element: expected {0} was {1}", //$NON-NLS-1$
				expectedType.getName(), semanticElement.eClass().getInstanceClassName()),
				expectedType.isInstance(semanticElement));
	}

	/**
	 * Returns the expected number of associated semantic elements.
	 *
	 * @return the expected number of associated semantic elements.
	 */
	public int getExpectedAssociatedElements() {
		return this.expectedAssociatedElements;
	}

	/**
	 * Sets the expected number of associated semantic elements.
	 *
	 * @param value
	 *            the expected number of associated semantic elements.
	 */
	public void setExpectedAssociatedElements(int value) {
		expectedAssociatedElements = value;
	}

	/**
	 * Returns the expected number of additional created elements in the checked {@code containmentFeature}.
	 *
	 * @return the expected number of additional created elements in the checked {@code containmentFeature}.
	 */
	public int getNumberOfExpectedCreatedElement() {
		return expectedCreatedElements;
	}

	/**
	 * Sets the expected number of additional created elements in the checked {@code containmentFeature}.
	 *
	 * @param value
	 *            the expected number of additional created elements in the checked {@code containmentFeature}.
	 */
	public void setExpectedCreatedElements(int value) {
		expectedCreatedElements = value;
	}

	/**
	 * Verifies the number of element after applying 'undo' action.
	 */

	@Override
	public void validateAfterUndo() {
		Assert.assertEquals("Unexpected elements after after undo.", //$NON-NLS-1$
				nbChildren, getContainmentFeatureValue().size());
	}

	/**
	 * Verifies the number of element after applying 'redo' action.
	 */
	@Override
	public void validateAfterRedo() {
		Assert.assertEquals("Unexpected additional elements after redo.", //$NON-NLS-1$
				getNumberOfExpectedCreatedElement(), getContainmentFeatureValue().size() - nbChildren);
	}

	/**
	 * Sets the index to get Semantic Element from representation.
	 * <p>
	 * If value is {@code -1}, the 'target' reference is used, otherwise
	 * the index is used on 'semanticElements' reference.
	 * </p>
	 *
	 * @param semanticIndex
	 *            the semanticIndex to set
	 */
	public void setSemanticIndex(int index) {
		this.semanticIndex = index;
	}

	/**
	 * Returns values of containment feature.
	 *
	 * @return
	 *         the value of the feature that should contain the created element
	 */
	protected final Collection<?> getContainmentFeatureValue() {
		return ((Collection<?>) semanticOwner.eGet(getContainmentFeature()));
	}

	/**
	 * Returns the expected containment feature.
	 *
	 * @return
	 *         the containment feature of the created element
	 */
	protected EReference getContainmentFeature() {
		return this.containmentFeature;
	}
}
